{%- if cookiecutter.use_postgresql or cookiecutter.use_sqlite -%}
"""
Cleanup old or stale data from the database.

This command is useful for maintenance tasks.
"""

import asyncio
from datetime import datetime, timedelta

import click

from app.commands import command, info, success, warning


@command("cleanup", help="Clean up old data from the database")
@click.option("--days", "-d", default=30, type=int, help="Delete records older than N days")
@click.option("--dry-run", is_flag=True, help="Show what would be deleted without making changes")
@click.option("--force", "-f", is_flag=True, help="Skip confirmation prompt")
def cleanup(days: int, dry_run: bool, force: bool) -> None:
    """
    Remove old records from the database.

    Example:
        project cmd cleanup --days 90
        project cmd cleanup --days 30 --dry-run
        project cmd cleanup --days 7 --force
    """
    cutoff_date = datetime.utcnow() - timedelta(days=days)

    if dry_run:
        info(f"[DRY RUN] Would delete records older than {cutoff_date}")
        return

    if not force and not click.confirm(f"Delete all records older than {days} days ({cutoff_date})?"):
        warning("Aborted.")
        return

{%- if cookiecutter.use_postgresql %}
    from app.db.session import async_session_maker

    async def _cleanup():
        async with async_session_maker() as _session:
            info(f"Cleaning up records older than {cutoff_date}...")

            # Add your cleanup logic here
            # Example:
            # result = await session.execute(
            #     delete(YourModel).where(YourModel.created_at < cutoff_date)
            # )
            # await session.commit()
            # deleted_count = result.rowcount

            deleted_count = 0  # Replace with actual count
            success(f"Deleted {deleted_count} records.")

    asyncio.run(_cleanup())
{%- elif cookiecutter.use_sqlite %}
    from app.db.session import SessionLocal

    with SessionLocal() as _session:
        info(f"Cleaning up records older than {cutoff_date}...")

        # Add your cleanup logic here
        # Example:
        # result = session.execute(
        #     delete(YourModel).where(YourModel.created_at < cutoff_date)
        # )
        # session.commit()
        # deleted_count = result.rowcount

        deleted_count = 0  # Replace with actual count
        success(f"Deleted {deleted_count} records.")
{%- endif %}
{%- endif %}
